// AcMapOAddCmdSaveBlocks.cpp: implementation of the AcMapOAddCmdSaveBlocks class.
//
// (C) Copyright 2001 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.

//
// CREATED BY:
// David Barbour, March 2001
//
// DESCRIPTION:
// 	Implementation of class AcMapOAddCmdSaveBlocks
//  This function loads blocks to Oracle

#include "stdafx.h"
#include "stdarx.h"
#include "resource.h"

#include "AdMapOracleConnection.h" // for AcMapOSEConnection
#include "AcMapOAddCmdSaveBlocks.h"	// for class AcMapOReactorSaveBlocks
#include "dbsymutl.h"	// for AcDbSymbolUtilities
#include "dbsymtb.h"	// for AcDbBlockTableRecord
#include "dbents.h"		// for AcDbBlockReference
#include "dbcfilrs.h"	// for AcDbWblockCloneFiler
#include "acutmem.h"	// for acutDelString
#include "oracl.h"
#include <iostream.h>
#include <fstream.h>
#include <tchar.h>


static AcMapOReactorSaveBlocks *pReactor = NULL;

//*****************************************************************************
// AcMapOAddCmdSaveBlock
// This function creates a reactor which watches export and
// saves each new block as it is encountered.             
void AcMapOAddCmdSaveBlock()
{

	AcMapOSEConnection * pConnect = AcMapOSEGetConnection();
	assert(pConnect);
	if(!pConnect->IsConnected())
	{
		acutPrintf("Not connected to Oracle.  Please connect before issuing this command\n");
		return;
	}
	if(pReactor)
	{
		acutPrintf("Already have a reactor in place.");
		return;
	}

	acutPrintf("REGISTERING REACTOR\n");
	pReactor = new AcMapOReactorSaveBlocks();
	pConnect->AddConnectionReactor(pReactor);
	pConnect->AddExportReactor(pReactor);
	
	acutPrintf("READY TO SAVE BLOCKS\n");
	
}

// AcMapOAddCmdStopSaveBlock
// This function removes and deletes the reactor
void AcMapOAddCmdStopSaveBlock()
{

	AcMapOSEConnection * pConnect = AcMapOSEGetConnection();
	assert(pConnect);
	acutPrintf("REMOVING REACTOR\n");
	if(pReactor)
	{
		pConnect->RemoveConnectionReactor(pReactor);
		pConnect->RemoveExportReactor(pReactor);
	}
	delete pReactor;
	pReactor = NULL;
	
	acutPrintf("DONE REMOVING REACTOR\n");
	
}

///////////////////////////////////////////////////////////////////////////////
// Construction/Destruction
///////////////////////////////////////////////////////////////////////////////

AcMapOReactorSaveBlocks::AcMapOReactorSaveBlocks()
	: m_pConnection(NULL),
	m_pDb(NULL)
	
{

	m_pConnection = AcMapOSEGetConnection();
	assert(m_pConnection);
	m_oDatabase = m_pConnection->Database();
}


AcMapOReactorSaveBlocks::~AcMapOReactorSaveBlocks()
{
	//void
}	

// Connection reactor overrides
// If schema or connection changes, then some of the stored information
// may be incorrect so this information is cleaned.
void
AcMapOReactorSaveBlocks::SchemaChanged()
{
	Clean();
}

void
AcMapOReactorSaveBlocks::Disconnected()
{
	Clean();
}

// Export reactor overrides
// This function breaks in at the point where we can determine the
// AcDbObjectId and the Oracle Id for each object.  This is necessary
// since objects being exported typically do not have any Oracle 
// identification.  Objects being updated do, however.
void
AcMapOReactorSaveBlocks::ObjectCached(
	AcDbEntity *pEnt,
	OValue oracleID)
{
	// sanity checking.  Would imply that this function can be called
	// twice with the same oracle id before the map is cleared.
	long lOracleID=(long)oracleID;
	assert(m_mapOra2AcadIds.end() == m_mapOra2AcadIds.find(lOracleID));
	AcDbObjectId objId = pEnt->objectId();
	m_mapOra2AcadIds[lOracleID] = objId;	
}	


// This function takes each exported object, checks if it is a block 
// reference (insert), and processes it.
void
AcMapOReactorSaveBlocks::ObjectsExported(std::vector<OValue> &vOracleIDs)
{
	// get current database every call since we 
	// can't tell if this has changed.
	m_pDb = acdbHostApplicationServices()->workingDatabase();
	assert(m_pDb);
	// iterate thru each exported object.
	std::vector<OValue>::const_iterator itend = vOracleIDs.end();
	for(std::vector<OValue>::const_iterator it = vOracleIDs.begin();
		it != itend;
		++it)
	{
		unsigned long orclId = (long)*it;
		MapOracleId2AcadId::const_iterator iterObjId = m_mapOra2AcadIds.find(orclId);
		if(iterObjId == m_mapOra2AcadIds.end())
		{
			// just some sanity checking.  This should not fail since
			// ObjectCached should be called once for each of the objects
			// in this function parameter vector.
			assert(false);
			continue;
		}
		// get the AcDbObjectId from the Oracle identifier
		AcDbObjectId objId = iterObjId->second;
		AcDbObject *pObj;
 		if(Acad::eOk == acdbOpenAcDbObject(pObj, objId, AcDb::kForRead))
 		{
			AcDbBlockReference *pBlockRef = AcDbBlockReference::cast(pObj);;
			if(NULL == pBlockRef)
			{
				// its not a block reference.
				pObj->close();
				continue;
			}
			assert(pBlockRef);
 			HandleBlockRef(pBlockRef);
			pBlockRef->close();
 		}
	}
	// we have finished with this set of entities, prepare for the 
	// next set.
	m_mapOra2AcadIds.clear();
}	



// private functions
// remove stored information as it is no longer valid in the new
// schema or connection
void
AcMapOReactorSaveBlocks::Clean()
{
	m_savedBlocks.clear();
	m_mapOra2AcadIds.clear();
}	


// HandleBlockRef
// This function checks to see if we have handled this block before
// and stores it to the Oracle database if we haven't.
// There is no checking to see if it is the same block as previously
// handled.  Different blocks should have different names.
void
AcMapOReactorSaveBlocks::HandleBlockRef(AcDbBlockReference *pBlockRef)
{
	assert(m_pDb);
	assert(pBlockRef);
	if(!pBlockRef) return;
	
	AcDbObjectId blockId = pBlockRef->blockTableRecord();
	AcDbObject *pObj;
 	if(Acad::eOk != acdbOpenAcDbObject(pObj, blockId, AcDb::kForRead))
 	{
		assert(false);
		return;
 	}
	AcDbBlockTableRecord *pBlock = AcDbBlockTableRecord::cast(pObj);
	if(NULL == pBlock)
	{
		// not a block.
		pObj->close();
		return;
	}

	char *pName;
 	if(Acad::eOk != AcDbSymbolUtilities::getSymbolName(pName, blockId))
 	{
		// sanity checking
		assert(false);
		pBlock->close();
		return;
 	}
	
	// check to see if the table exists yet.  Create it if not.	
	if(!CheckAndCreateTable())
	{
		acutPrintf(
			"Could not create table SAMPLE_BLOCKS in schema %s.\n", 
			m_pConnection->Schema());
		pBlock->close();
		acutDelString(pName);
		return;
	}
	if(IsBlockAlreadySaved(pName))
	{
		pBlock->close();
		acutDelString(pName);
		return;
	}
	SaveBlock(pName, pBlock, blockId);
	acutPrintf("Saved block %s.\n", pName);
	pBlock->close();
	acutDelString(pName);
	
}	


// CheckAndCreateTable
// This function creates the SAMPLE_BLOCKS table if it does not
// already exist.
bool
AcMapOReactorSaveBlocks::CheckAndCreateTable()
{
	std::string strSql("Select 1 from ");
	strSql += m_pConnection->Schema();
	strSql += ".SAMPLE_BLOCKS";
	ODynaset ds;
	oresult ores=ds.Open(
		m_oDatabase, 
		strSql.c_str(),
		ODYNASET_READONLY|ODYNASET_NOCACHE);
	if (OSUCCESS!=ores)
	{
		// need to create table
		strSql = "Create table ";
		strSql += m_pConnection->Schema();
		strSql += ".SAMPLE_BLOCKS (BlockName VARCHAR2(255), BlockGeom BLOB)";
		return (OSUCCESS == m_oDatabase.ExecuteSQL(strSql.c_str()));
	}
	return true;
}	


// IsBlockAlreadySaved
// This function first looks for the block in a locally stored
// set.  Since there are likely to be many block references for
// each block, it is time consuming to check the Oracle table
// each time.  If the block name is not in our set, then we check
// the Oracle table.  If its there we assume its the same as our 
// block, add its name to the set and return true.  If not in
// the Oracle table we return false.
bool
AcMapOReactorSaveBlocks::IsBlockAlreadySaved(const char *pName)
{
	StringSet::const_iterator it = m_savedBlocks.find(pName);
	
	if(it != m_savedBlocks.end())
	{
		// this block is in our list of previously encountered blocks.
		return true;
	}
	// check the table for this block
	std::string strSql = "Select * from ";
	strSql += m_pConnection->Schema();
	strSql += ".SAMPLE_BLOCKS where UPPER(BlockName) = UPPER(\'";
	strSql += pName;
	strSql += "\')";
	ODynaset ds;
	if(OSUCCESS != ds.Open(
		m_oDatabase,
		strSql.c_str(),
		ODYNASET_NOCACHE|ODYNASET_NOBIND|ODYNASET_READONLY))
	{
		// something went wrong.  This typically indicates that
		// the table doesn't exist, but we should have created it
		// before getting here.
		assert(false);
		return false;
	}
	// GetRowCount can be a time consuming task, but here
	// we know that there are only 1 or 0 rows.
	if(0 == ds.GetRecordCount())
	{
		return false;
	}
	m_savedBlocks.insert(pName);
	return true;
}	

// SaveBlock
// This function saves the block to the table.  
// Preconditions:
// The table exists, the block is open for read, the block has not been 
// saved to the table previously.
bool
AcMapOReactorSaveBlocks::SaveBlock(
	const char *pName,
	AcDbBlockTableRecord *pBlock,
	AcDbObjectId& blockId)
{
	oresult ores;
	std::string strSql = "Insert into ";
	strSql += m_pConnection->Schema();
	strSql += ".SAMPLE_BLOCKS values (\'";
	strSql += pName;
	strSql += "\', EMPTY_BLOB())";
	ores = m_oDatabase.ExecuteSQL(strSql.c_str());
	OConnection oConnection = m_oDatabase.GetConnection();

	std::string strQuery("Select * from ");
	strQuery += m_pConnection->Schema();
	strQuery += ".SAMPLE_BLOCKS where BlockName = \'";
	strQuery += pName;
	strQuery += "\'";
	ODynaset dynaset;
	ores = dynaset.Open(m_oDatabase,
		strQuery.c_str());
	if(OSUCCESS != ores)
	{
		acutPrintf("Cannot open dynaset on table SAMPLE_BLOCKS.\n");
		return false;
	}
	int bOpen = dynaset.IsOpen();
	int nCount = dynaset.GetRecordCount();

	OBlob oBlob;
	ores = dynaset.StartEdit();  // starts edit operation
	
	char tempPath[_MAX_PATH];
	char fullPath[_MAX_PATH];
	char drive[_MAX_PATH];
	char dir[_MAX_PATH];

	::GetTempPath(_MAX_PATH, tempPath);
	_tsplitpath(tempPath, drive, dir, NULL, NULL);
	_tmakepath(fullPath, drive, dir, pName, NULL);
	AcDbDatabase *pBlockDb = new AcDbDatabase;
	Acad::ErrorStatus es = m_pDb->wblock(pBlockDb, blockId);
	es = pBlockDb->saveAs(fullPath);
	delete pBlockDb;

	_tmakepath(fullPath, drive, dir, pName, "dwg");
	// find the size
	//open file and get file size
	fstream fs;
	fs.open(fullPath, ios::in);
	assert(fs.is_open());
	
	int retval = fs.setmode(filebuf::binary);
	fs.seekg(0, ios::end);
	unsigned long nRemainLen = fs.tellg();
	fs.seekg(0, ios::beg);

	assert(nRemainLen > 0);
	assert(OSUCCESS == ores);
	// chunks must be less than 64k since size is an unsigned short.
	unsigned char chunkp[CHUNK_LEN];
	unsigned long nBytesRead;
	ores = dynaset.GetFieldValue("BlockGeom", &oBlob);
	assert(OSUCCESS == ores);
	bool bNull = oBlob.IsNull();
	//unsigned long optchunk = oBlob.GetOptimumChunkSize();
	unsigned char chPieceType = OLOB_FIRST_PIECE;

	if (nRemainLen <= CHUNK_LEN)
	{
		chPieceType = OLOB_ONE_PIECE;
	}
	else
	{
		oBlob.EnableStreaming(nRemainLen);
	}


	try
	{

		while(nRemainLen > 0)
		{
			//len = (nRemainLen < CHUNK_LEN) ? nRemainLen : CHUNK_LEN;
			fs.read(chunkp, CHUNK_LEN);
			nBytesRead = fs.gcount();

			int nBytesWritten = oBlob.Write(chunkp, nBytesRead, chPieceType);
			assert(nBytesWritten == nBytesRead);
			nRemainLen -= nBytesRead;
			chPieceType = (nRemainLen <= CHUNK_LEN) ? OLOB_LAST_PIECE : OLOB_NEXT_PIECE;
		}
	}
	
	catch(OException &oExc)
	{
		const char *pText = oExc.GetErrorText();
	}
	// clean up
	fs.close();
	BOOL bDeleted = ::DeleteFile(fullPath);
	oBlob.DisableStreaming();
	ores = dynaset.Update();
	if(OSUCCESS == ores)
	{
		acutPrintf("Block %s written to table.\n", pName);
		m_savedBlocks.insert(pName);
		return true;
	}
	else
	{
		acutPrintf("Block %s could not be written to table.\n", pName);
		return false;
	}
}	



// This function provides the less operator needed by STL
// (Standard Template Library) associative classes
bool
AcMapOReactorSaveBlocks::less_ignore_case::operator() (
    const std::string& s1,
    const std::string& s2) const
{
	// this is a globally aware version of stricmp()
    return _tcsicmp(s1.c_str(), s2.c_str()) < 0;
}


// eof